// Copyright 2021 yuzu Emulator Project
// Licensed under GPLv2 or any later version
// Refer to the license.txt file included.

#pragma once

#include <functional>
#include <mutex>
#include <unordered_map>

#include "common/input.h"
#include "common/param_package.h"
#include "common/point.h"
#include "common/quaternion.h"
#include "common/settings.h"
#include "common/vector_math.h"
#include "core/hid/hid_types.h"
#include "core/hid/motion_input.h"

namespace Core::HID {

struct ConsoleMotionInfo {
    Input::MotionStatus raw_status;
    MotionInput emulated{};
};

using ConsoleMotionDevices = std::unique_ptr<Input::InputDevice>;
using TouchDevices = std::array<std::unique_ptr<Input::InputDevice>, 16>;

using ConsoleMotionParams = Common::ParamPackage;
using TouchParams = std::array<Common::ParamPackage, 16>;

using ConsoleMotionValues = ConsoleMotionInfo;
using TouchValues = std::array<Input::TouchStatus, 16>;

struct TouchFinger {
    u64_le last_touch{};
    Common::Point<float> position{};
    u32_le id{};
    bool pressed{};
    TouchAttribute attribute{};
};

// Contains all motion related data that is used on the services
struct ConsoleMotion {
    bool is_at_rest{};
    Common::Vec3f accel{};
    Common::Vec3f gyro{};
    Common::Vec3f rotation{};
    std::array<Common::Vec3f, 3> orientation{};
    Common::Quaternion<f32> quaternion{};
};

using TouchFingerState = std::array<TouchFinger, 16>;

struct ConsoleStatus {
    // Data from input_common
    ConsoleMotionValues motion_values{};
    TouchValues touch_values{};

    // Data for HID services
    ConsoleMotion motion_state{};
    TouchFingerState touch_state{};
};

enum class ConsoleTriggerType {
    Motion,
    Touch,
    All,
};

struct ConsoleUpdateCallback {
    std::function<void(ConsoleTriggerType)> on_change;
};

class EmulatedConsole {
public:
    /**
     * Contains all input data related to the console like motion and touch input
     */
    EmulatedConsole();
    ~EmulatedConsole();

    YUZU_NON_COPYABLE(EmulatedConsole);
    YUZU_NON_MOVEABLE(EmulatedConsole);

    /// Removes all callbacks created from input devices
    void UnloadInput();

    /// Sets the emulated console into configuring mode. Locking all HID service events from being
    /// moddified
    void EnableConfiguration();

    /// Returns the emulated console to the normal behaivour
    void DisableConfiguration();

    /// Returns true if the emulated console is on configuring mode
    bool IsConfiguring() const;

    /// Reload all input devices
    void ReloadInput();

    /// Overrides current mapped devices with the stored configuration and reloads all input devices
    void ReloadFromSettings();

    /// Saves the current mapped configuration
    void SaveCurrentConfig();

    /// Reverts any mapped changes made that weren't saved
    void RestoreConfig();

    // Returns the current mapped motion device
    Common::ParamPackage GetMotionParam() const;

    /**
     * Updates the current mapped motion device
     * @param ParamPackage with controller data to be mapped
     */
    void SetMotionParam(Common::ParamPackage param);

    /// Returns the latest status of motion input from the console with parameters
    ConsoleMotionValues GetMotionValues() const;

    /// Returns the latest status of touch input from the console with parameters
    TouchValues GetTouchValues() const;

    /// Returns the latest status of motion input from the console
    ConsoleMotion GetMotion() const;

    /// Returns the latest status of touch input from the console
    TouchFingerState GetTouch() const;

    /**
     * Adds a callback to the list of events
     * @param ConsoleUpdateCallback that will be triggered
     * @return an unique key corresponding to the callback index in the list
     */
    int SetCallback(ConsoleUpdateCallback update_callback);

    /**
     * Removes a callback from the list stopping any future events to this object
     * @param Key corresponding to the callback index in the list
     */
    void DeleteCallback(int key);

private:
    /**
     * Updates the motion status of the console
     * @param A CallbackStatus containing gyro and accelerometer data
     */
    void SetMotion(Input::CallbackStatus callback);

    /**
     * Updates the touch status of the console
     * @param callback: A CallbackStatus containing the touch position
     * @param index: Finger ID to be updated
     */
    void SetTouch(Input::CallbackStatus callback, std::size_t index);

    /**
     * Triggers a callback that something has changed on the console status
     * @param Input type of the event to trigger
     */
    void TriggerOnChange(ConsoleTriggerType type);

    bool is_configuring{false};
    f32 motion_sensitivity{0.01f};

    ConsoleMotionParams motion_params;
    TouchParams touch_params;

    ConsoleMotionDevices motion_devices;
    TouchDevices touch_devices;

    mutable std::mutex mutex;
    std::unordered_map<int, ConsoleUpdateCallback> callback_list;
    int last_callback_key = 0;

    // Stores the current status of all console input
    ConsoleStatus console;
};

} // namespace Core::HID
