package main

import (
	"fmt"
	"github.com/antchfx/xmlquery"
	"os"
	"path/filepath"
	"regexp"
	"strings"
)

type dependencyItem struct {
	pkgName, envName, targetNetVer string
}

// Parse csproj file, and get list of packages to download
func ParseDependencies(csprojPath string) (dependencies []dependencyItem, err error) {
	f, err := os.Open(csprojPath)
	if err != nil {
		return
	}
	parsed, err := xmlquery.Parse(f)
	if err != nil {
		return
	}
	references := xmlquery.Find(parsed, "//Project/ItemGroup/Reference")
	for _, ref := range references {
		// This codeblock processes one <Reference>
		// println("DEBUG: " + ref.OutputXML(true))
		pkgName, envName, targetNetVer := "", "", ""
		for _, attr := range ref.Attr {
			if attr.Name.Local == "Include" {
				pkgName = attr.Value // GOT
			}
		}
		if pkgName != "" {
			subNode := ref.SelectElement("HintPath")
			if subNode != nil {
				hintPath := subNode.OutputXML(false)
				endPos := strings.Index(hintPath, ")")
				envName = hintPath[2:endPos] // GOT

				if hintPath[endPos+2:endPos+5] == "lib" {
					hintPath = hintPath[endPos+6:]
					endPos := strings.IndexAny(hintPath, "/\\") // Maybe `/lib/` or `\lib\`
					targetNetVer = hintPath[:endPos]
				}
			}
		}
		dependencies = append(dependencies, dependencyItem{pkgName, envName, targetNetVer})
	}
	return
}

func ScanCsprojInDirectory(dirPath string) (csprojPathes []string, err error) {
	libRegEx, err := regexp.Compile("^.+\\.(csproj)$")
	if err != nil {
		return
	}

	err = filepath.Walk(dirPath, func(path string, info os.FileInfo, err error) error {
		if err == nil && libRegEx.MatchString(info.Name()) {
			csprojPathes = append(csprojPathes, path)
		}
		return nil
	})

	return
}

// This function panic on error, because it's not recoverable.
// Returns key-value pairs for environment variables to set.
func GenerateCorextEnvvar(localRepoPath, buildOutputPath string, deps []dependencyItem) map[string]string {
	result := make(map[string]string)

	csprojPluginTemplate := "<Project xmlns=\"http://schemas.microsoft.com/developer/msbuild/2003\"><PropertyGroup><OutputPath>%v</OutputPath></PropertyGroup></Project>"
	fname, err := stringToTempFile(fmt.Sprintf(csprojPluginTemplate, buildOutputPath))
	panicErrorIfAny(err, "CreateTempFile")
	result["EnvironmentConfig"] = fname

	for _, dep := range deps {
		if dep.envName != "" {
			pkgPath, err := GetPackagePathFromName(localRepoPath, dep.pkgName)
			logErrorIfAny(err, "GetPackagePathFromName: ")
			result[dep.envName] = pkgPath
		}
	}

	return result
}
