package chocopy.pa3;

import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Stack;

import chocopy.common.analysis.SymbolTable;
import chocopy.common.analysis.AbstractNodeAnalyzer;
import chocopy.common.analysis.types.SymbolType;
import chocopy.common.analysis.types.ValueType;
import chocopy.common.astnodes.*;
import chocopy.common.codegen.*;

import static chocopy.common.codegen.RiscVBackend.Register.*;

/**
 * This is where the main implementation of PA3 will live.
 *
 * A large part of the functionality has already been implemented
 * in the base class, CodeGenBase. Make sure to read through that
 * class, since you will want to use many of its fields
 * and utility methods in this class when emitting code.
 *
 * Also read the PDF spec for details on what the base class does and
 * what APIs it exposes for its sub-class (this one). Of particular
 * importance is knowing what all the SymbolInfo classes contain.
 */
public class CodeGenImpl extends CodeGenBase {
    private class BetterRsicVBackend {
        private final RiscVBackend backend;

        BetterRsicVBackend(RiscVBackend _backend) {
            backend = _backend;
        }

        public void emitNoop(String comment) {
            if (comment != null) {
                backend.emitMV(ZERO, ZERO, "Noop: " + comment);
            }
        }

        public void emitPush(RiscVBackend.Register reg, String comment) {
            backend.emitADDI(SP, SP, -1 * backend.getWordSize(), comment);
            backend.emitSW(reg, SP, 0, comment);
        }

        public void emitPop(RiscVBackend.Register reg, String comment) {
            if (reg != null)
                backend.emitLW(reg, SP, 0, comment);
            backend.emitADDI(SP, SP, backend.getWordSize(), comment);
        }

        public void emitCall(Label calledLabel, String comment) {
            // Arguments should be already pushed to stack.
            backend.emitJAL(calledLabel, comment);
        }

        public void emitFunctionBegin(String funcName, String comment) {
            emitNoop(comment);
            emitPush(RA, "backup my return address.");
            emitPush(FP, "backup parent fp.");
            backend.emitADDI(FP, SP, 2 * backend.getWordSize(), "Change FP to apply new frame.");

            backend.emitJ(new Label("$" + funcName + "$saveReg"), "Jump to save callee-saved registers");
            backend.emitLocalLabel(new Label("$" + funcName + "$saveRegContinue"), "Begin real function codes:");
        }

        public void emitFunctionEnd(String funcName, Void regMgr, String comment) {
            // Return value should be in A0
            List<RiscVBackend.Register> savedRegs = new ArrayList<RiscVBackend.Register>() {}; // regMgr.registerToSaveAndRestoreInFunc;

            emitNoop(comment);
            backend.emitADDI(SP, FP, -(2 + savedRegs.size()) * backend.getWordSize(), "Revert all local variables on this dying frame.");
            for (int i = savedRegs.size(); i-- > 0; ) {
                emitPop(savedRegs.get(i), "restore callee-saved reg.");
            }
            emitPop(FP, "restore parent FP.");
            emitPop(RA, "restore my return address.");
            backend.emitJR(RA, "Real return!");

            backend.emitLocalLabel(new Label("$" + funcName + "$saveReg"), "Save callee-saved registers before use.");
            for (int i = 0; i < savedRegs.size(); ++i) {
                emitPush(savedRegs.get(i), "save callee-saved reg.");
            }
            backend.emitJ(new Label("$" + funcName + "$saveRegContinue"), "go back to execute real function code.");
        }

        public void emitPushIntVal(RiscVBackend.Register tmpReg, Integer val, String comment) {
            emitNoop(comment);
            backend.emitLI(tmpReg, val, "INT VAL");
            emitPush(tmpReg, null);
            backend.emitLA(tmpReg, intClass.getDispatchTableLabel(), "OBJECT HEAD - DISPATCH TBL");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 4, "OBJECT HEAD - SIZE = 3+1");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 1, "OBJECT HEAD - TYPE = INT");
            emitPush(tmpReg, null);
            backend.emitMV(tmpReg, SP, "Return INT address.");
        }

        public void emitPushBoolVal(RiscVBackend.Register tmpReg, Boolean val, String comment) {
            emitNoop(comment);
            backend.emitLI(tmpReg, val ? 1 : 0, "BOOL VAL");
            emitPush(tmpReg, null);
            backend.emitLA(tmpReg, boolClass.getDispatchTableLabel(), "OBJECT HEAD - DISPATCH TBL");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 4, "OBJECT HEAD - SIZE = 3+1");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 2, "OBJECT HEAD - TYPE = BOOL");
            emitPush(tmpReg, null);
            backend.emitMV(tmpReg, SP, "Return BOOL address.");
        }

        public void emitPushStrVal(RiscVBackend.Register tmpReg, String val, String comment) {
            emitNoop(comment);
            byte[] bytes = val.getBytes(StandardCharsets.US_ASCII);
            int dataLen = bytes.length / backend.getWordSize() + 1; // Equals to ceil((len+1) DIVIDE WORD_SIZE)

            for (int cter = 0; cter < dataLen; ++cter) {
                int myRangeBegin = (dataLen - cter - 1) * backend.getWordSize();
                int myRangeEnd = Math.min(myRangeBegin + backend.getWordSize(), bytes.length);
                int curr = 0;
                assert backend.getWordSize() == 4; // curr should be 4 byte = int
                String debug_pushed_str = "";
                for (int shift = 0; shift < myRangeEnd - myRangeBegin; ++shift) {
                    curr += bytes[myRangeBegin + shift] << (shift * 8);
                    debug_pushed_str += Integer.toString(bytes[myRangeBegin + shift]) + " ";
                }

                backend.emitLI(tmpReg, curr, "STR VAL - " + debug_pushed_str);
                emitPush(tmpReg, "STR VAL - PUSH A BLOCK");
            }

            backend.emitLI(tmpReg, bytes.length, "STR ATTR - __len__");
            emitPush(tmpReg, null);
            backend.emitLA(tmpReg, strClass.getDispatchTableLabel(), "OBJECT HEAD - DISPATCH TBL");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 3 + dataLen + 1, "OBJECT HEAD - SIZE");
            emitPush(tmpReg, null);
            backend.emitLI(tmpReg, 3, "OBJECT HEAD - TYPE = STR");
            emitPush(tmpReg, null);
            backend.emitMV(tmpReg, SP, "Return STR OBJ address.");
        }
    }


    /** A code generator emitting instructions to BACKEND. */
    public CodeGenImpl(RiscVBackend backend) {
        super(backend);
        betterBackend = new BetterRsicVBackend(backend);
    }

    private BetterRsicVBackend betterBackend;

    /** Operation on None. */
    private final Label errorNone = new Label("error.None");
    /** Division by zero. */
    private final Label errorDiv = new Label("error.Div");
    /** Index out of bounds. */
    private final Label errorOob = new Label("error.OOB");

    /**
     * Emits the top level of the program.
     *
     * This method is invoked exactly once, and is surrounded
     * by some boilerplate code that: (1) initializes the heap
     * before the top-level begins and (2) exits after the top-level
     * ends.
     *
     * You only need to generate code for statements.
     *
     * @param statements top level statements
     */
    protected void emitTopLevel(List<Stmt> statements) {
        StmtAnalyzer stmtAnalyzer = new StmtAnalyzer(null);
        backend.emitADDI(SP, SP, -2 * backend.getWordSize(),
                         "Saved FP and saved RA (unused at top level).");
        backend.emitSW(ZERO, SP, 0, "Top saved FP is 0.");
        backend.emitSW(ZERO, SP, 4, "Top saved RA is 0.");
        backend.emitADDI(FP, SP, 2 * backend.getWordSize(),
                         "Set FP to previous SP.");
        for (Stmt stmt : statements) {
            stmt.dispatch(stmtAnalyzer);
        }
        backend.emitLI(A0, EXIT_ECALL, "Code for ecall: exit");
        backend.emitEcall(null);
    }

    /**
     * Emits the code for a function described by FUNCINFO.
     *
     * This method is invoked once per function and method definition.
     * At the code generation stage, nested functions are emitted as
     * separate functions of their own. So if function `bar` is nested within
     * function `foo`, you only emit `foo`'s code for `foo` and only emit
     * `bar`'s code for `bar`.
     */
    protected void emitUserDefinedFunction(FuncInfo funcInfo) {
        backend.emitGlobalLabel(funcInfo.getCodeLabel());
        StmtAnalyzer stmtAnalyzer = new StmtAnalyzer(funcInfo);
        // Prologue:
        backend.emitSW(FP, SP, 0, "Save old FP");
        backend.emitSW(RA, SP, -4, "Save RA");
        backend.emitADDI(SP, SP, -8, "Increment stack pointer 2 places");
        backend.emitADDI(FP, SP, 4, "FP is one slot below top of stack");

        // Emit code for function stmts
        for (Stmt stmt : funcInfo.getStatements()) {
            stmt.dispatch(stmtAnalyzer);
        }

        backend.emitMV(A0, ZERO, "Returning None implicitly");
        backend.emitLocalLabel(stmtAnalyzer.epilogue, "Epilogue");

        // FIXME: {... reset fp etc. ...}
        backend.emitLW(RA, FP, 0, "Restore RA");
        backend.emitLW(FP, FP, 4, "Restore old FP");
        backend.emitJR(RA, "Return to caller");
    }

    /** An analyzer that encapsulates code generation for statments. */
    private class StmtAnalyzer extends AbstractNodeAnalyzer<Void> {
        /*
         * The symbol table has all the info you need to determine
         * what a given identifier 'x' in the current scope is. You can
         * use it as follows:
         *   SymbolInfo x = sym.get("x");
         *
         * A SymbolInfo can be one the following:
         * - ClassInfo: a descriptor for classes
         * - FuncInfo: a descriptor for functions/methods
         * - AttrInfo: a descriptor for attributes
         * - GlobalVarInfo: a descriptor for global variables
         * - StackVarInfo: a descriptor for variables allocated on the stack,
         *      such as locals and parameters
         *
         * Since the input program is assumed to be semantically
         * valid and well-typed at this stage, you can always assume that
         * the symbol table contains valid information. For example, in
         * an expression `foo()` you KNOW that sym.get("foo") will either be
         * a FuncInfo or ClassInfo, but not any of the other infos
         * and never null.
         *
         * The symbol table in funcInfo has already been populated in
         * the base class: CodeGenBase. You do not need to add anything to
         * the symbol table. Simply query it with an identifier name to
         * get a descriptor for a function, class, variable, etc.
         *
         * The symbol table also maps nonlocal and global vars, so you
         * only need to lookup one symbol table and it will fetch the
         * appropriate info for the var that is currently in scope.
         */

        /** Symbol table for my statements. */
        private SymbolTable<SymbolInfo> sym;

        /** Label of code that exits from procedure. */
        protected Label epilogue;

        /** The descriptor for the current function, or null at the top
         *  level. */
        private FuncInfo funcInfo;

        private int _r_local_label_counter = 427;

        private Void allocPrototype(SymbolType typ, RiscVBackend.Register objReg) {
            // Convention: if typ is INT_TYPE or BOOL_TYPE then objReg contains the literal value, not a pointer.
            if (typ.equals(SymbolType.INT_TYPE)) {
                backend.emitLA(A0, intClass.getPrototypeLabel(), "Load prototype address");
                // save our objReg on stack before `alloc`
                backend.emitADDI(SP, SP, -4, "Decrement stack ptr 1");
                backend.emitSW(objReg, SP, 0,  "Store the literal value");
                backend.emitJAL(objectAllocLabel, "Allocate int");
                backend.emitLW(objReg, SP, 0, "Load literal value again");
                backend.emitADDI(SP, SP, 4, "Increment stack ptr 1");
                backend.emitSW(objReg, A0, 3 * backend.getWordSize(), "Set __int__ value");
            } else if (typ.equals(SymbolType.STR_TYPE)) {
                // objReg already has a ptr in this case. just move to A0.
                backend.emitMV(A0, objReg, "move string ptr to A0");
            } else if (typ.equals(SymbolType.BOOL_TYPE)){
                backend.emitLA(A0, constants.getBoolConstant(false), "Load FALSE bool");
                backend.emitSLLI(objReg, objReg, 4, "Compute offset to correct object");
                backend.emitADD(A0, A0, objReg, "Add offset to curr pointer");
            }
            return null;
        }

        /** An analyzer for the function described by FUNCINFO0, which is null
         *  for the top level. */
        StmtAnalyzer(FuncInfo funcInfo0) {
            funcInfo = funcInfo0;
            if (funcInfo == null) {
                sym = globalSymbols;
            } else {
                sym = funcInfo.getSymbolTable();
            }
            epilogue = generateLocalLabel();
        }

        // FIXME: Example of statement.
        @Override
        public Void analyze(ReturnStmt stmt) {
            stmt.value.dispatch(this);
            backend.emitLW(A0, SP, 0, "Load return value into A0");
            backend.emitJ(epilogue, "Jump to epilogue");
            return null;
        }

        @Override
        public Void analyze(ExprStmt exprStmt) {
            exprStmt.expr.dispatch(this);
            return null;
        }

        @Override
        public Void analyze(AssignStmt node) {
            backend.emitADDI(SP, SP, -4, "Increment stack ptr by 1 for Condition");
            node.value.dispatch(this);
            RiscVBackend.Register tmpReg = /*regMgr.borrowOneTmp()*/ T6;
            RiscVBackend.Register tmpReg2 = /*regMgr.borrowOneTmp()*/ T5;
            betterBackend.emitPop(tmpReg, "ASSIGN: Get the value to assign: Pop the expr result from stack");

            SymbolType symbolType = null;
            for(Expr expr : node.targets) {
                if(! (expr instanceof Identifier)) {
                    throw new RuntimeException("assignment target should be Identifier.");
                }
                SymbolInfo symbolInfo = sym.get(((Identifier) expr).name);
                Label label = null;
                if(symbolType == null) {
                    // first run:
                    if (symbolInfo instanceof GlobalVarInfo) {
                        GlobalVarInfo globalVarInfo = (GlobalVarInfo) symbolInfo;
                        symbolType = globalVarInfo.getVarType();
                    } else if (symbolInfo instanceof StackVarInfo) {
                        StackVarInfo stackVarInfo = (StackVarInfo) symbolInfo;
                        symbolType = stackVarInfo.getVarType();
                    }
                    if (symbolType.equals(SymbolType.INT_TYPE) || symbolType.equals(SymbolType.BOOL_TYPE)) {
                        // first run:
                        /* All int and bool variable is not wrapped.
                        backend.emitLW(tmpReg, tmpReg, 3 * backend.getWordSize(), "ASSIGN: Load the value of INT or BOOL");
                        */
                    }
                }

                // do assign
                if (symbolInfo instanceof GlobalVarInfo) {
                    GlobalVarInfo globalVarInfo = (GlobalVarInfo) symbolInfo;
                    symbolType = globalVarInfo.getVarType();
                    backend.emitSW(tmpReg, globalVarInfo.getLabel(), tmpReg2, "ASSIGN: Assign to global var");
                } else if (symbolInfo instanceof StackVarInfo) {
                    StackVarInfo stackVarInfo = (StackVarInfo) symbolInfo;
                    symbolType = stackVarInfo.getVarType();
                    throw new RuntimeException("ASSIGN to stack variable: NOT IMPLEMENTED: StackVarInfo not implemented.");
                }
            }
            return null;
        }

        private String getLablePrefix(String name) {
            String labelPrefix = "";
            if(funcInfo != null) {
                labelPrefix += "$" + funcInfo.getFuncName();
            }
            labelPrefix += "$" + name + _r_local_label_counter;
            ++_r_local_label_counter;
            return labelPrefix;
        }

        @Override
        public Void analyze(IfStmt node) {
            backend.emitADDI(SP, SP, -4, "Increment stack ptr by 1 for Condition");
            node.condition.dispatch(this);
            RiscVBackend.Register tmpReg = /*regMgr.borrowTmp()*/ T6;
            betterBackend.emitPop(tmpReg, "IfStmt: Get condition");

            String labelPrefix = getLablePrefix("if");

            Label elseLabel = new Label( labelPrefix + ".else");
            Label exitLabel = new Label( labelPrefix + ".exit");
            backend.emitBEQ(ZERO, tmpReg, elseLabel, "IfStmt: If condition is false, jump to else.");
            for(Stmt stmt : node.thenBody) {
                stmt.dispatch(this);
            }
            backend.emitJ(exitLabel, "IfStmt: THEN body done. jump to end.");
            backend.emitLocalLabel(elseLabel, "IfStmt: ELSE");
            for(Stmt stmt : node.elseBody) {
                stmt.dispatch(this);
            }
            backend.emitLocalLabel(exitLabel, "IfStmt: DONE");

            return null;
        }

        @Override
        public Void analyze(WhileStmt node) {
            String labelPrefix = getLablePrefix("while");

            Label beginLabel = new Label(labelPrefix + ".begin");
            Label exitLabel = new Label(labelPrefix + ".exit");

            backend.emitLocalLabel(beginLabel, "WhileStmt: Begin");
            backend.emitADDI(SP, SP, -4, "Increment stack ptr by 1 for Condition");
            node.condition.dispatch(this);
            RiscVBackend.Register tmpReg = /*borrowOneTmp*/ T6;
            betterBackend.emitPop(tmpReg, "GetCondition");
            backend.emitBEQ(ZERO, tmpReg, exitLabel, "WhileSTmt, IfFalse, exit");

            for(Stmt stmt : node.body) {
                stmt.dispatch(this);
            }
            backend.emitJ(beginLabel, "WhileStmt: GOTO begin");

            backend.emitLocalLabel(exitLabel, "WhileStmt: End");
            return null;
        }


        @Override
        public Void analyze(CallExpr callExpr) {
            // NOT DONE
            if (globalSymbols.get(callExpr.function.name) instanceof FuncInfo) {
                FuncInfo func =  (FuncInfo) globalSymbols.get(callExpr.function.name);
                for (Expr e : callExpr.args) {
                    backend.emitADDI(SP, SP, -4, "Increment stack ptr by 1");
                    e.dispatch(this);
                }
                // Push parameters onto stack
                if (func.getFuncName().equals("print")) {
                    backend.emitLW(T0, SP, 0, "Load param for boxing");
                    allocPrototype(callExpr.args.get(0).getInferredType(), T0);
                    // after this A0 should contain address of newly allocated object
                    backend.emitSW(A0, SP, 0, "put boxed argument back on stack");
                }
                backend.emitJAL(func.getCodeLabel(), "Invoke function");

                // Restore SP
                for (Expr e : callExpr.args) {
                    backend.emitADDI(SP, SP, 4, "Decrement stack ptr by 1");
                }

            }
            return null;
        }

        @Override
        public Void analyze(BooleanLiteral literal) {
            /* Push boolean literal onto stack, incrementing stack pointer*/
            backend.emitLI(A0, (literal.value == true) ? 1 : 0, "Load boolean literal into a0");
            backend.emitSW(A0, SP, 0, "Push on stack");
            return null;
        }

        @Override
        public Void analyze(IntegerLiteral literal) {
            backend.emitLI(A0, literal.value, "Load integer literal into a0");
            backend.emitSW(A0, SP, 0, "Push on stack");
            return null;
        }

        @Override
        public Void analyze(StringLiteral literal) {
            backend.emitLA(A0, constants.getStrConstant(literal.value), "Load string addr into a0");
            backend.emitSW(A0, SP, 0, "Push on stack");
            return null;
        }

        @Override
        public Void analyze(Identifier id) {
            // NOT DONE
            boolean argFlag = false;
            int argCtr = 0;
            if (funcInfo != null) {
                for (String s : funcInfo.getParams()) {
                    if (id.name.equals(s)) {
                        argFlag = true;
                        break;
                    }
                    argCtr++;
                }
            }
            SymbolInfo ident = sym.get(id.name);
            if (argFlag){
                int offset = argCtr * backend.getWordSize();
                backend.emitLW(A0, FP, offset, "Load stack argument");
            } else if (ident instanceof GlobalVarInfo) {     // global var case
                backend.emitLW(A0, ((GlobalVarInfo) ident).getLabel(), "Load identifier");
                backend.emitSW(A0, SP, 0, "Push to stack");
            } else if (ident instanceof StackVarInfo){ // local variable
                //TODO: Figure out how we can keep track of whether the var has already been pushed to stack.
                StackVarInfo identVar = (StackVarInfo) ident;
                if (id.getInferredType().equals(SymbolType.INT_TYPE)) {
                    IntegerLiteral value = (IntegerLiteral) (identVar.getInitialValue());
                    backend.emitLI(A0, value.value, "Load initial value of StackVar");
                    backend.emitSW(A0, SP, 0, "Push to stack");
                }
            }
            return null;
        }

        @Override
        public Void analyze(BinaryExpr binaryExpr) {
            switch (binaryExpr.operator) {
                case "+":
                    if (binaryExpr.left.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitADD(T0, T0, T1, "Add operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "-":
                    if (binaryExpr.getInferredType().equals(SymbolType.INT_TYPE)) {    
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        
                        backend.emitSUB(T0, T1, T0, "Subtract operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "*":
                    if (binaryExpr.getInferredType().equals(SymbolType.INT_TYPE)) {    
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        
                        backend.emitMUL(T0, T1, T0, "Subtract operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "//":
                    if (binaryExpr.getInferredType().equals(SymbolType.INT_TYPE)) {    
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        
                        backend.emitDIV(T0, T1, T0, "Subtract operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "%":
                    if (binaryExpr.getInferredType().equals(SymbolType.INT_TYPE)) {    
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        
                        backend.emitREM(T0, T1, T0, "Subtract operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "==":
                    if (binaryExpr.getInferredType().equals(SymbolType.BOOL_TYPE) ||
                            binaryExpr.getInferredType().equals(SymbolType.INT_TYPE) ) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitXOR(T0, T0, T1, "== operator on BOOL/INT");
                        backend.emitSEQZ(T0, T0, "Set to 1 if XOR results in 0, they were equal");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }

                case "!=":
                    if (binaryExpr.getInferredType().equals(SymbolType.BOOL_TYPE) ||
                            binaryExpr.getInferredType().equals(SymbolType.INT_TYPE) ) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitXOR(T0, T0, T1, "!= operator on BOOL/INT");
                        backend.emitSNEZ(T0, T0, "Set to 1 if XOR was not 0, they were not equal");
                        // Don't think we need to do the SEQZ/SNEZ instructions here because XOR will already result in 1
                        // if they are not equal, which is the desired result.
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "<=":
                    if (binaryExpr.left.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitADDI(T1, T1, -1, "decrement LEFT by 1");
                        backend.emitSLT(T1, T1, T0, "LEFT <=? RIGHT");
                        backend.emitSW(T1, SP, 0, "Push result onto stack");
                        return null;
                    }
                case ">=":
                    if (binaryExpr.left.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitADDI(T0, T0, -1, "decrement RIGHT by 1");
                        backend.emitSLT(T1, T0, T1, "LEFT >=? RIGHT");
                        backend.emitSW(T1, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "<":
                    if (binaryExpr.left.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitSLT(T1, T1, T0, "LEFT <? RIGHT");
                        backend.emitSW(T1, SP, 0, "Push result onto stack");
                        return null;
                    }
                case ">":
                    if (binaryExpr.left.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitSLT(T1, T0, T1, "LEFT >? RIGHT");
                        backend.emitSW(T1, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "and":
                    if (binaryExpr.getInferredType().equals(SymbolType.BOOL_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitLW(T1, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        Label skipSecond = generateLocalLabel();
                        backend.emitBEQZ(T1, skipSecond, "If operand 1 is FALSE, don't eval second");
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitAND(T1, T1, T0, "AND operands");
                        backend.emitLocalLabel(skipSecond, "After second operand");
                        backend.emitSW(T1, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "or":
                    if (binaryExpr.getInferredType().equals(SymbolType.BOOL_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        binaryExpr.left.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 1 into t1");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        Label skipSecond = generateLocalLabel();
                        backend.emitBNEZ(T0, skipSecond, "If first is not zero (it's true), skip second");
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for second operand");
                        binaryExpr.right.dispatch(this);
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for second operand");
                        backend.emitOR(T0, T1, T0, "OR operands");
                        backend.emitLocalLabel(skipSecond, "Skip past second operand");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
            }

            return null;
        }

        @Override
        public Void analyze(UnaryExpr unaryExpr) {
            switch (unaryExpr.operator) {
                case "-":
                    if (unaryExpr.getInferredType().equals(SymbolType.INT_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        unaryExpr.operand.dispatch(this);
                 
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        
                        backend.emitSUB(T0, ZERO, T0, "Subtract operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
                case "not":
                    if (unaryExpr.getInferredType().equals(SymbolType.BOOL_TYPE)) {
                        backend.emitADDI(SP, SP, -4, "Increment stack ptr for first operand");
                        unaryExpr.operand.dispatch(this);
                 
                        backend.emitLW(T0, SP, 0, "Load operand 2 into t0");
                        backend.emitADDI(SP, SP, 4, "Decrement stack ptr for first operand");
                        
                        backend.emitSEQZ(T0, T0, "NOT operands");
                        backend.emitSW(T0, SP, 0, "Push result onto stack");
                        return null;
                    }
            }
            return null;
        }


        // FIXME: More, of course.

    }

    /**
     * Emits custom code in the CODE segment.
     *
     * This method is called after emitting the top level and the
     * function bodies for each function.
     *
     * You can use this method to emit anything you want outside of the
     * top level or functions, e.g. custom routines that you may want to
     * call from within your code to do common tasks. This is not strictly
     * needed. You might not modify this at all and still complete
     * the assignment.
     *
     * To start you off, here is an implementation of three routines that
     * will be commonly needed from within the code you will generate
     * for statements.
     *
     * The routines are error handlers for operations on None, index out
     * of bounds, and division by zero. They never return to their caller.
     * Just jump to one of these routines to throw an error and
     * exit the program. For example, to throw an OOB error:
     *   backend.emitJ(errorOob, "Go to out-of-bounds error and abort");
     *
     */
    protected void emitCustomCode() {
//        backend.emitGlobalLabel(new Label("int-alloc"));
//        backend.emitLA(A0, intClass.getPrototypeLabel(), "Get int prototype address");
//        backend.emitJAL(objectAllocLabel, "Allocate int");
//
//        FuncInfo allocInt = makeFuncInfo("int-alloc", 0, SymbolType.OBJECT_TYPE,
//                globalSymbols, null, this::emitStdFunc);
//        allocInt.addParam(makeStackVarInfo("arg", SymbolType.OBJECT_TYPE,
//                null, allocInt));
//        functions.add(allocInt);
//        System.out.println(allocInt.getBaseName());
//        globalSymbols.put(allocInt.getBaseName(), allocInt);
        emitErrorFunc(errorNone, "Operation on None");
        emitErrorFunc(errorDiv, "Divison by zero");
        emitErrorFunc(errorOob, "Index out of bounds");
    }

    /** Emit an error routine labeled ERRLABEL that aborts with message MSG. */
    private void emitErrorFunc(Label errLabel, String msg) {
        backend.emitGlobalLabel(errLabel);
        backend.emitLI(A0, ERROR_NONE, "Exit code for: " + msg);
        backend.emitLA(A1, constants.getStrConstant(msg),
                       "Load error message as str");
        backend.emitADDI(A1, A1, getAttrOffset(strClass, "__str__"),
                         "Load address of attribute __str__");
        backend.emitJ(abortLabel, "Abort");
    }
}
