/*
  * This program is free software: you can redistribute it and/or modify
  * it under the terms of the GNU General Public License as published by
  * the Free Software Foundation, either version 3 of the License, or
  * any later version.
  *
  * This program is distributed in the hope that it will be useful,
  * but WITHOUT ANY WARRANTY; without even the implied warranty of
  * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
  * GNU General Public License for more details.
  *
  * You should have received a copy of the GNU General Public License
  * along with this program.  If not, see <http://www.gnu.org/licenses/>.
  *
  * Additional permission under GNU GPL version 3 section 7
  *
  * If you modify this Program, or any covered work, by linking or combining
  * it with OpenSSL (or a modified version of that library), containing parts
  * covered by the terms of OpenSSL License and SSLeay License, the licensors
  * of this Program grant you additional permission to convey the resulting work.
  *
  */

#include <assert.h>
#include <cmath>
#include <chrono>
#include <cstring>
#include <thread>
#include <bitset>

#include "IBackend.hpp"
#include "BackendConnector.hpp"

#include "cpu/minethd.h"
#ifndef CONF_NO_CUDA
#	include "nvidia/minethd.h"
#endif
#ifndef CONF_NO_OPENCL
#	include "amd/minethd.h"
#endif
#include "miner_work.h"
#include "GlobalStates.hpp"
 #include <cstdlib>

#include "Plugin.hpp"
#include "../Environment.hpp"
#include "../console.h"
#include "../Params.hpp"

namespace xmrstak
{


bool BackendConnector::self_test()
{
	
	return true;
}

std::vector<IBackend*>* BackendConnector::thread_starter(miner_work& pWork)
{
    GlobalStates::inst().iGlobalJobNo = 0;
	GlobalStates::inst().iConsumeCnt = 0;


	std::vector<IBackend*>* pvThreads = new std::vector<IBackend*>;

#ifndef CONF_NO_CUDA
	if(Params::inst().useNVIDIA)
	{
		Plugin nvidiaPlugin("NVIDIA", "xmrstak_cuda_backend");
		std::vector<IBackend*>* nvidiaThreads = nvidiaPlugin.startBackend(static_cast<uint32_t>(pvThreads->size()), pWork, Environment::inst());
		pvThreads->insert(std::end(*pvThreads), std::begin(*nvidiaThreads), std::end(*nvidiaThreads));
		if(nvidiaThreads->size() == 0)
			printer::inst()->print_msg(L0, "WARNING: backend NVIDIA disabled.");
	}
#endif

#ifndef CONF_NO_OPENCL
	if(Params::inst().useAMD)
	{
		Plugin amdPlugin("AMD", "xmrstak_opencl_backend");
		std::vector<IBackend*>* amdThreads = amdPlugin.startBackend(static_cast<uint32_t>(pvThreads->size()), pWork, Environment::inst());
		pvThreads->insert(std::end(*pvThreads), std::begin(*amdThreads), std::end(*amdThreads));
		if(amdThreads->size() == 0)
			printer::inst()->print_msg(L0, "WARNING: backend AMD disabled.");
	}
#endif

#ifndef CONF_NO_CPU
	if(Params::inst().useCPU)
	{
		auto cpuThreads = cpu::minethd::thread_starter(static_cast<uint32_t>(pvThreads->size()), pWork);
		pvThreads->insert(std::end(*pvThreads), std::begin(cpuThreads), std::end(cpuThreads));
		if(cpuThreads.size() == 0)
			printer::inst()->print_msg(L0, "WARNING: backend CPU disabled.");
	}
#endif
	
	GlobalStates::inst().iThreadCount = pvThreads->size();
	return pvThreads;
}

} // namepsace xmrstak
